function [d_stks,psf] = deconv_vstk(varargin)
%little function that applies a 2d deconvolution to a virtual stack of images.
%Synatax:   [d_stk,psf] = deconv_vstk('edgetaper',1,'deconv','reg','numit',10,...
%               'initpsf',10);
%Input:     'edgetaper' = taper the image edges according to the PSF.
%               Default = 0; off.
%           'deconv' = type of deconvolution used.  Default = lucy
%               Lucy-Richardson.  Other types: 'reg' = regularized,
%               'wiener' = wiener, and 'blind' = blind, based on maximum
%               likelihood algorithm.
%           'numit' = number of iteration for deconvolution, default = 10.
%               only applicable to blind and Lucy-Richardson.
%           'dampar' = the threshold deviation of the resulting image from
%               the original image (in terms of the standard deviation of
%               Poisson noise) below which damping occurs
%           'subsmpl' =  denotes subsampling and is used when the PSF is
%               given on a grid that is SUBSMPL times finer than the image.
%               Default is 1.
%           'initpsf' = the size of the initial psf generated for the blind
%               deconvolution.  The PSF restoration is affected strongly by
%               the size of the initial guess INITPSF and less by the
%               values it contains.
%           'cfx' = change file data types to preserve flux ratios. Default
%               = 0, off.  In this case, a 8 bit image will go to 16 and
%               back to 8bit, 16 bit image will go to 32bit then back to 16
%               for storage.
%           'save' = save the stacks as tiff files automatically. Default =
%               1.  0 = off.
%           'bksub' = background subtract.  Default = 0 (off).
%           'fullout' = determins whether to output the background images
%               generated during background subtraction or not.  Default =
%               0 (off).
%           'dir' = the root directory of the images
%           'psfdir' = the directory of PSFs
%Output:    d_stk = the deconvoluted image stack
%           psf = the PSF used.
%           fac = the psf parameter used. (only if lucy is used)

[initpsf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,cfx,bk,dir_tmp,psf_dir,fullout] = parse(varargin);

%now lets get the directories with the files
if isempty(dir_tmp) %if no image root directory was entered
    prompt_box('title','Select Image Directory','prompt1','Open the root folder of your images','position','center');
    pause(0.25);
    dir_tmp = uigetdir2('','Directory where the images are located');    %get the directory
end

%now get the directory of the PSF files
if isempty(psf_dir)     %if no psf directory was entered
    prompt_box('title','Select PSF Directory','prompt1','Open the folder of your PSFs','position','center');
    pause(0.25);
    psf_dir = uigetdir3('','Directory where the PSFs are located');    %get the directory
end
    
%get the psf filenames and parse it first
psf_files = dir(psf_dir);
%truncate the first two rows which are . and ..
psf_files = {psf_files.name};
%grab only the wave length numbers
strmask = isstrprop(psf_files,'digit');
for i = 3:size(psf_files,2)
    psf_wave(i) = str2double(psf_files{i}(strmask{i}));
end

%now grab look in the root directory to see how many directories are there
ch_dir = dir(dir_tmp);
%Here things will bifricate, if the root has directories then all the files
%in the root will be ignored, but if there are no directories in the root,
%then this will go into file deconvolution mode.
dirfile = [ch_dir.isdir];   %grab all of the isdir numbers
dirfile = max(dirfile(3:end));    %create the switch

%create the root deconv directory
mkdir(dir_tmp,'deconv');
if bk   %make background subtracted folders
    mkdir(dir_tmp,'bksubtracted');
    mkdir(dir_tmp,'background');
end

switch dirfile
    case 1  %directory mode
        %grab the directory/file names
        dirnames = {ch_dir.name};
        %get the root metat data files
        file_tmp = dirnames(~[ch_dir.isdir]);     %only the filenames are taken
        dirnames = dirnames([ch_dir.isdir]);     %only the directories please
        %grab the metadata files *.txt
        root_meta = findfiletype(file_tmp,'.txt');  %only txts
        %get the wavelength names from the directories
        %dirn_strmsk = isstrprop(dirnames,'digit');
        dir_wave = regexp(dirnames,'\d\d\d','match');   %grab only the numbers that are 3 in a row as the wavelength
        %parse the directory structure to see which PSF to apply
        h = waitbar(0,['Deconvolving Directory:',dirnames{3}],'position',[10 50 275 50]);
        for j = 3:size(dirnames,2)    %go through the directories and files, except the first two again
            waitbar((j-2)/(size(dirnames,2)-2),h,['Deconvolving directory: ',dirnames{j}]);   %update progress
            %curr_wave = str2num(dirnames{j}(dirn_strmsk{j}));    %grab the current wavelength
            try
                curr_wave = str2num(cell2mat(dir_wave{j}));
                %now lets see if there is a match
                decon_wave = psf_wave==curr_wave;
            catch   %no numbers in the directory name
                decon_wave = 0;
            end
            %PSF is needed
            if ~strcmp(deconv_type,'blind')          %if the deconvolution is not blind open PSF
                if max(decon_wave)    %there is a match go go go
                    %lets open the psf
                    psf = imread([psf_dir,filesep,psf_files{decon_wave==1}]);
                else    %no match kick out to user
                    prompt_box('title','Import the PSF','prompt1','Select PSF for folder:',...
                        'prompt2',dirnames{j},'position','center');
                    pause(0.25)
                    [psf_filename,psf_path] = uigetfile2b({'*.tif','Tif files (*.tif)';'*.tiff','Tiff Files (*.tiff)';...
                        '*.*','All Files';},'Open PSF','Multiselect','off');
                    psf = imread([psf_path,psf_filename]);
                end
                psf = im2double(psf);       %the PSF needs to be in double percision.
                %matlab expect the PSF to sum to 1, thus lets do a
                %little math to make sure
                psf = psf./(sum(sum(psf)));
            else                                %if the deconvolution is blind generate PSF
                psf = fspecial('gaussian',initpsf,10);
            end
            %now grab all of the files in that directory
            curr_dir = dir([dir_tmp,filesep,dirnames{j}]);  
            file_tmp = {curr_dir.name};     %grab the names
            file_tmp = file_tmp(~[curr_dir.isdir]);     %only the filenames are taken
            %grab the metadata files *.txt
            metafiles = findfiletype(file_tmp,'.txt');  %only txts
            %we only want .tif files
            tifloc = strfind(file_tmp,'.tif');   %find the tiff files
            tifidx = zeros(size(tifloc));     %initialize/reset
            for m = 1:size(tifloc,2)
                if isempty(tifloc{m})   %if this is not a tiff file
                    tifidx(m) = 0;  %ignore
                else
                    tifidx(m) = 1;  %keep
                end
            end
            file_tmp = file_tmp(logical(tifidx));   %remove non-tif files                    
            mkdir([dir_tmp,filesep,'deconv'],dirnames{j});
            if bk   %make background subtracted folders
                mkdir([dir_tmp,filesep,'bksubtracted'],dirnames{j});
                mkdir([dir_tmp,filesep,'background'],dirnames{j});
            end
            h2 = waitbar(0,['Deconvolving Slice: 1']);
            for k = 1:size(file_tmp,2)  %go through the files
                curr_img = imread([dir_tmp,filesep,dirnames{j},filesep,file_tmp{k}]);
                if bk
                    [curr_img,bk_img] = bksub('img',curr_img,'sav',0,'se_type','ball','r',20,'h',20,'n',8,'fullout',fullout);
                    sub_img = curr_img;     %store subtracted image for output
                end
                %process the image info
                imgclass = class(curr_img);      %get image type             
                switch imgclass
                    case 'uint8'
                        dampar = uint8(dampar);      %the image is unsigned 8bit
                        if cfx      %preserve flux
                            curr_img = uint16(curr_img);
                            imgclass = 'uint16';
                            dampar = uint16(dampar);
                        end
                    case 'uint16'
                        dampar = uint16(dampar);     %the image is unsigned 16bit
                        if cfx
                            curr_img = im2double2(uint32(curr_img));
                            imgclass = 'uint32b';
                            dampar = double(dampar);
                        end
                    case 'uint32'                    %no support for 32bit yet, convert to double
                        %dampar = uint32(dampar);
                        curr_img = im2double2(curr_img);
                        dampar = double(dampar);
                    case 'double'
                        dampar = double(dampar);     %the image is double
                    case 'single'
                        dampar = single(dampar);     %the image is single
                end
                %now let the fun begin
                if edgetapered        %if the edges need to be tapered
                    curr_img = edgetaper(curr_img,psf);
                end
                %now apply the deconvolution
                switch deconv_type
                    case 'lucy'
                        curr_img = deconvlucy(curr_img,psf,numit,dampar,[],[],subsmpl);
                    case 'reg'
                        curr_img = deconvreg(curr_img,psf);
                    case 'wiener'
                        curr_img = deconvwnr(curr_img,psf);
                    case 'blind'
                        initpsf = psf;      %psf is the output so it will change, temp the PSF.
                        [curr_img,psf] = deconvblind(curr_img,initpsf,numit,dampar);
                end
                %stretch dynamic range (decomment if you want it on)
                %stk_out = imnorm(stk_out);
                %make the cfx output folder if necessary
                if cfx&&strcmp('uint32b',imgclass)&&k==1
                    mkdir([dir_tmp,filesep,'deconv'],[dirnames{j},'_16bit']);
                end 
                %if cfx is on change the data back to the initial data type
                if cfx
                    switch imgclass
                        case 'uint32'                    %32bit in 32bit out
                            curr_img = uint32(curr_img.*4294967295);
                        case 'uint32b'                    %16bit in 16bit out a special case I guess
                            curr_img2 = im2uint16(imnorm(curr_img));          %output the data in 16bit for compatibility and ease of use
                            curr_img = uint32(curr_img.*4294967295);      %output the full image
                        otherwise
                            curr_img = im2uint16(curr_img);   %uint16 is default
                    end
                else  %otherwise make sure the output is uint16 or unint32 (comment out if you do not want this)
                    switch imgclass
                        case 'uint32'                    %32bit in 32bit out
                            curr_img = uint32(curr_img.*4294967295);
                        otherwise
                            curr_img = im2uint16(curr_img);   %uint16 is default
                    end
                end
                %now output the file
                %save the stacks automatically if selected.
                if sav
                    stk2tiff(curr_img',file_tmp{k},[dir_tmp,filesep,'deconv',filesep,dirnames{j},filesep]);
                    if cfx&&strcmp('uint32b',imgclass)
                        stk2tiff(curr_img2',file_tmp{k},[dir_tmp,filesep,'deconv',filesep,dirnames{j},'_16bit',filesep]);
                    end
                    if bk   %save out the background subtracted data
                        stk2tiff(sub_img',file_tmp{k},[dir_tmp,filesep,'bksubtracted',filesep,dirnames{j},filesep]);
                        if fullout
                            %now save out each image stack
                            stk2tiff(bk_img',file_tmp{k},[dir_tmp,filesep,'background',filesep,dirnames{j},filesep]);
                        end
                    end
                end
                waitbar(k/size(file_tmp,2),h2,['Deconvolving Slice: ',num2str(k+1)]);   %update progress
            end
            close(h2)
            %copy over the metadata
            if sav
                for m = 1:size(metafiles,2)
                    copyfile([dir_tmp,filesep,dirnames{j},filesep,metafiles{m}],[dir_tmp,filesep,'deconv',filesep,dirnames{j},filesep,metafiles{m}]);
                end
            end
        end
        %copy over the root metadata
        if sav
            for m = 1:size(root_meta,2)
                copyfile([dir_tmp,filesep,root_meta{m}],[dir_tmp,filesep,'deconv',filesep,root_meta{m}]);
            end
        end
otherwise   %file mode
    %grab the directory/file names
    file_tmp = {ch_dir.name};
    file_tmp = file_tmp(~[ch_dir.isdir]);     %only the filenames are taken
    %grab the metadata files *.txt
    metafiles = findfiletype(file_tmp,'.txt');  %only txts
    %we only want .tif files
    tifloc = strfind(file_tmp,'.tif');   %find the tiff files
    tifidx = zeros(size(tifloc));     %initialize/reset
    for m = 1:size(tifloc,2)
        if isempty(tifloc{m})   %if this is not a tiff file
            tifidx(m) = 0;  %ignore
        else
            tifidx(m) = 1;  %keep
        end
    end
    file_tmp = file_tmp(logical(tifidx));   %remove non-tif files
    %get the wavelength names from the root directory
    %grab only the last directory of the root
    dirmsk = isstrprop(dir_tmp,'punct');    %find the punctuations
    dirpunct = dir_tmp(dirmsk);             %grab the punctuations
    punct_idx = find(dirmsk==1);            %find the location of the punctuations
    %find the last directory demarcation
    for l = size(dirpunct,2):-1:1,
        if strcmp(dirpunct(l),filesep)
            break
        end
    end
    %grab the last folder
    curr_wave = dir_tmp(punct_idx(l):end);
    %wave_strmsk = isstrprop(curr_wave,'digit');
    %grab the wavelength from root folder
    %curr_wave = str2double(curr_wave(wave_strmsk));    %grab the current wavelength
    dir_wave = regexp(curr_wave,'\d\d\d','match');   %grab only the numbers that are 3 in a row as the wavelength
    try
        curr_wave = str2num(cell2mat(dir_wave));
        %process PSF
        decon_wave = psf_wave==curr_wave;
    catch   %no numbers in directory name
        decon_wave = 0;
    end
    %PSF is needed
    if ~strcmp(deconv_type,'blind')          %if the deconvolution is not blind open PSF
        if max(decon_wave)    %there is a match go go go
            %lets open the psf
            psf = imread([psf_dir,filesep,psf_files{decon_wave==1}]);
        else    %no match kick out to user
            prompt_box('title','Import the PSF','prompt1','Select PSF for folder:',...
                'prompt2',dir_tmp(punct_idx(l):end),'position','center');
            pause(0.25)
            [psf_filename,psf_path] = uigetfile2b({'*.tif','Tif files (*.tif)';'*.tiff','Tiff Files (*.tiff)';...
                '*.*','All Files';},'Open PSF','Multiselect','off');
            psf = imread([psf_path,psf_filename]);
        end
        psf = im2double(psf);       %the PSF needs to be in double percision.
        %matlab expect the PSF to sum to 1, thus lets do a
        %little math to make sure
        psf = psf./(sum(sum(psf)));
    else                                %if the deconvolution is blind generate PSF
        psf = fspecial('gaussian',initpsf,10);
    end
    %now go through the files
    h = waitbar(0,['Deconvolving Slice: 1']);
    for j = 1:size(file_tmp,2)
        curr_img = imread([dir_tmp,filesep,file_tmp{j}]);
        if bk
            [curr_img,bk_img] = bksub('img',curr_img,'sav',0,'se_type','ball','r',20,'h',20,'n',8,'fullout',fullout);
            sub_img = curr_img;     %store subtracted image for output
        end
        %process the image info
        imgclass = class(curr_img);      %get image type
        switch imgclass
            case 'uint8'
                dampar = uint8(dampar);      %the image is unsigned 8bit
                if cfx      %preserve flux
                    curr_img = uint16(curr_img);
                    imgclass = 'uint16';
                    dampar = uint16(dampar);
                end
            case 'uint16'
                dampar = uint16(dampar);     %the image is unsigned 16bit
                if cfx
                    curr_img = im2double2(uint32(curr_img));
                    imgclass = 'uint32b';
                    dampar = double(dampar);
                end
            case 'uint32'                    %no support for 32bit yet, convert to double
                %dampar = uint32(dampar);
                curr_img = im2double2(curr_img);
                dampar = double(dampar);
            case 'double'
                dampar = double(dampar);     %the image is double
            case 'single'
                dampar = single(dampar);     %the image is single
        end
        %now let the fun begin
        if edgetapered        %if the edges need to be tapered
            curr_img = edgetaper(curr_img,psf);
        end
        %now apply the deconvolution
        switch deconv_type
            case 'lucy'
                curr_img = deconvlucy(curr_img,psf,numit,dampar,[],[],subsmpl);
            case 'reg'
                curr_img = deconvreg(curr_img,psf);
            case 'wiener'
                curr_img = deconvwnr(curr_img,psf);
            case 'blind'
                initpsf = psf;      %psf is the output so it will change, temp the PSF.
                [curr_img,psf] = deconvblind(curr_img,initpsf,numit,dampar);
        end
        %stretch dynamic range (decomment if you want it on)
        %stk_out = imnorm(stk_out);
        %create the channel directory in deconv root
        if cfx&&strcmp('uint32b',imgclass)&&j==1
            mkdir(dir_tmp,'deconv_16bit');
        end
        %if cfx is on change the data back to the initial data type
        if cfx
            switch imgclass
                case 'uint32'                    %32bit in 32bit out
                    curr_img = uint32(curr_img.*4294967295);
                case 'uint32b'                    %16bit in 16bit out a special case I guess
                    curr_img2 = im2uint16(imnorm(curr_img));          %output the data in 16bit for compatibility and ease of use
                    curr_img = uint32(curr_img.*4294967295);      %output the full image
                otherwise
                    curr_img = im2uint16(curr_img);   %uint16 is default
            end
        else  %otherwise make sure the output is uint16 or unint32 (comment out if you do not want this)
            switch imgclass
                case 'uint32'                    %32bit in 32bit out
                    curr_img = uint32(curr_img.*4294967295);
                otherwise
                    curr_img = im2uint16(curr_img);   %uint16 is default
            end
        end
        %now output the file
        %save the stacks automatically if selected.
        if sav
            stk2tiff(curr_img',file_tmp{j},[dir_tmp,filesep,'deconv',filesep]);
            if cfx&&strcmp('uint32b',imgclass)
                stk2tiff(curr_img2',file_tmp{j},[dir_tmp,filesep,'deconv_16bit',filesep]);
            end
            if bk   %save out the background subtracted data
                stk2tiff(sub_img',file_tmp{j},[dir_tmp,filesep,'bksubtracted',filesep]);
                if fullout
                    %now save out each image stack
                    stk2tiff(bk_img',file_tmp{j},[dir_tmp,filesep,'background',filesep]);
                end
            end
        end
        waitbar(j/size(file_tmp,2),h,['Deconvolving Slice: ',num2str(j+1)]);   %update progress
    end
    %copy over the metadata
    if sav
        for m = 1:size(metafiles,2)
            copyfile([dir_tmp,filesep,metafiles{m}],[dir_tmp,filesep,'deconv',filesep,metafiles{m}]);
        end
    end
end
close(h)




%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [initpsf,edgetapered,deconv_type,numit,dampar,subsmpl,sav,cfx,bk,dir_tmp,psf_dir,fullout] = parse(input)

deconv_type = 'lucy';    %1=lucy-richardson(default), 2=regularized, 3=wiener, 4=blind.
edgetapered = 0;  %off.
numit = 10;     %default # of iteration = 10.
initpsf = 10;  %default = 10x10
sav = 1;       %default = save the file automatically
dampar = 0;     %default = 0 no dampling
subsmpl = 1;    %deault = 1 no subsampling
cfx = 0;     %by default off
bk = 0;      %background subtract off by default
dir_tmp = [];
psf_dir = [];
fullout = 0;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'edgetaper'
                    edgetapered = input{1,i+1};
                case 'deconv'
                    if ischar(input{1,i+1})
                        deconv_type = input{1,i+1};
                    else
                        warning(['Your entered deconvolution type is not recognized, reverting to defalut type.']);
                    end
                case 'numit'
                    numit = input{1,i+1};
                case 'save'
                    sav = input{1,i+1};
                case 'dampar'
                    dampar = input{1,i+1};
                case 'subsmpl'
                    subsmpl = input{1,i+1};
                case 'initpsf'
                    initpsf = input{1,i+1};
                case 'cfx'
                    cfx = input{1,i+1};
                case 'bksub'
                    bk = input{1,i+1};
                case 'dir'
                    dir_tmp = input{1,i+1};
                case 'psfdir'
                    psf_dir = input{1,i+1};
                case 'fullout'
                    fullout = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end